#ifndef __CFileInformation__
#define __CFileInformation__

#include "../Basics/CCountedObject.hpp"
#include "../Basics/CString.hpp"
#include "../Basics/CSystemString.hpp"
#include "CTime.hpp"

#ifndef WIN32
	#include <dirent.h>
#endif

//	===========================================================================

using Exponent::Host::CTime;
using Exponent::Basics::CCountedObject;
using Exponent::Basics::CString;
using Exponent::Basics::CSystemString;

//	===========================================================================

namespace Exponent
{
	namespace Host
	{
		/**
		 * @class CFileInformation CFileInformation.hpp
		 * @brief Stores information about a file
		 *
		 * @date 16/09/2004
		 * @author Paul Chana
		 * @version 1.0.0 Initial version
		 * @version 1.0.1 Added assignment operator
		 *
		 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
		 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
		 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
		 * All content is the Intellectual property of Exp Digital Uk.\n
		 * Certain sections of this code may come from other sources. They are credited where applicable.\n
		 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
		 *
		 * $Id: CFileInformation.hpp,v 1.7 2007/02/08 21:06:44 paul Exp $
		 */
		class CFileInformation : public CCountedObject
		{
			/** @cond */
			EXPONENT_CLASS_DECLARATION;
			/** @endcond */

//	===========================================================================

		public:

//	===========================================================================

			/**
			 * @enum EFileAttribute
			 * @brief Attributes of the files
			 */
			enum EFileAttribute
			{
				e_archive = 0,			/**< Archive @deprecated */
				e_compressed,			/**< Compressed (is this not the same as above??) @deprecated */
				e_directory,			/**< Directory */
				e_encryted,				/**< Encrypted @deprecated */
				e_hidden,				/**< Hidden @deprecated */
				e_normal,				/**< Normal file */
				e_offline,				/**< Offline file @deprecated */
				e_readOnly,				/**< Read only @deprecated */
				e_reparse,				/**< ?? Never did figure out what windows thinks this type is @deprecated */
				e_system,				/**< System file */
				e_temporary,			/**< Temp file @deprecated */
				e_unknown				/**< Unkown file type */
			};

//	===========================================================================

			/**
			 * Construction
			 */
			CFileInformation();

			/**
			 * Copy Construction
			 * @param info The info to copy
			 */
			CFileInformation(const CFileInformation &info);
			
/** @cond */
#ifdef WIN32
/** @endcond */
			/**
			 * Construction from a find data object
			 * @param fileInfo The window file info
			 */
			CFileInformation(WIN32_FIND_DATA &fileInfo);
			
/** @cond */
#else
/** @endcond */
			/**
			 * Construction from a dirent structure
			 * @param directoryEntry The unix dirent structure
			 */
			CFileInformation(dirent *directoryEntry);
			
/** @cond */
#endif
/** @endcond */

			/**
			 * Destruction
			 */
			virtual ~CFileInformation();

//	===========================================================================

			/**
			 * Assignment operator
			 * @param info The info to copy
			 * @retval CFileInformation& A reference to this
			 */
			virtual CFileInformation &operator = (const CFileInformation &info);

			/**
			 * Comparison for a qsort
			 * @param info1 The first info 
			 * @param info2 The second info
			 * @retval <0 info1 goes before info2 0 info1 == info2 >0 info1 goes after info2
			 */
			static int compareFiles(const CFileInformation **info1, const CFileInformation **info2);

//	===========================================================================
			
/** @cond */
#ifdef WIN32
/** @endcond */
			/**
			 * Set from a find data object
			 * @param fileInfo The window file info
			 */
			void setFileInformationFromData(WIN32_FIND_DATA &fileInfo);
			
/** @cond */
#endif
/** @endcond */

			/**
			 * Set the file attribute
			 * @param attribute The file atributes
			 */
			void setFileAttribute(const EFileAttribute attribute);

			/**
			 * Set the creation time
			 * @param creationTime The creation time
			 */
			void setCreationTime(const CTime &creationTime);

			/**
			 * Set the last access time
			 * @param lastAccess last access time
			 */
			void setLastAccessTime(const CTime &lastAccess);

			/**
			 * Set the last write time
			 * @param lastWrite The last write time
			 */
			void setLastWriteTime(const CTime &lastWrite);

			/**
			 * Set the size of the file in bytes
			 * @param size The size in bytes
			 */
			void setSizeInBytes(const double size);

			/**
			 * Set the file name
			 * @param filename the filename
			 */
			void setFilename(const CString &filename);

			/**
			 * Set the folders name
			 * @param foldername The containing folders name
			 */
			void setFoldername(const CString &foldername);

//	===========================================================================

			/**
			 * Get the file attribute
			 */
			EFileAttribute getFileAttribute() const { return m_attribute; }

			/**
			 * Get the creation time
			 * @retval const CTime& The creation time
			 */
			const CTime &getCreationTime() const { return m_creationTime; }

			/**
			 * Get the last access time
			 * @retval const CTime& The last access time
			 */
			const CTime &getLastAccessTime() const { return m_accessTime; }

			/**
			 * Get the last write time
			 * @retval const CTime& The last write time
			 */
			const CTime &getLastWriteTime() const { return m_writeTime; }

			/**
			 * Get the size of the file in bytes
			 * @retval double File size in bytes
			 */
			double getSizeInBytes() const { return m_fileSizeInBytes; }

			/**
			 * Get the file name
			 * @retval const CString& The filename
			 */
			const CString &getFilename() const { return m_filename; }

			/**
			 * Get the file name
			 * @retval const CString& The MSDOS style filename
			 */
			const CString &getShortFilename() const { return m_shortFilename; }

			/**
			 * Get the folder name
			 * @retval const CString& The containing folder name
			 */
			const CString &getFoldername() const { return m_foldername; }

			/**
			 * Get the full path
			 * @retval const CSystemString& The full path to the file
			 */
			const CSystemString &getFullPath() const { return m_fullPath; }

			/**
			 * Is this a directory
			 * @retval bool True if this points to a folder, false otherwise
			 */
			bool isDirectory() const { return m_attribute == e_directory; }

//	===========================================================================

		protected:

//	===========================================================================

			EFileAttribute m_attribute;				/**< File attribute */
			CTime m_creationTime;					/**< Time of creation */
			CTime m_accessTime;						/**< Last access time */
			CTime m_writeTime;						/**< Last write time */
			double m_fileSizeInBytes;				/**< File size in bytes */
			CString m_filename;						/**< Name of the file */
			CString m_shortFilename;				/**< MS-DOS style filename */
			CString m_foldername;					/**< Folder that contains it */
			CSystemString m_fullPath;				/**< Full path to the file */
		};
	}
}
#endif	// End of CFileInformation.hpp
